<?php

namespace FactelBundle\Controller;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use FactelBundle\Entity\Factura;
use FactelBundle\Entity\FacturaHasProducto;
use FactelBundle\Entity\Impuesto;
use FactelBundle\Entity\CampoAdicional;
use FactelBundle\Entity\CargaArchivo;
use FactelBundle\Entity\CargaError;
use FactelBundle\Form\FacturaType;
use JMS\SecurityExtraBundle\Annotation\Secure;
use Symfony\Component\HttpFoundation\Response;
use FactelBundle\Util;

require_once 'ProcesarComprobanteElectronico.php';
require_once 'reader.php';


/**
 * Factura controller.
 *
 * @Route("/comprobantes/factura")
 */
class FacturaController extends Controller
{

    /**
     * Lists all Emisor entities.
     *
     * @Route("/", name="factura")
     * @Method("GET")
     * @Template()
     */
    public function indexAction()
    {

        return array();
    }

    /**
     * Lists all Factura entities.
     *
     * @Route("/facturas", name="all_factura")
     * @Secure(roles="ROLE_EMISOR")
     * @Method("GET")
     */
    public function facturasAction()
    {
        if (isset($_GET['sEcho'])) {
            $sEcho = $_GET['sEcho'];
        }
        if (isset($_GET['iDisplayStart'])) {
            $iDisplayStart = intval($_GET['iDisplayStart']);
        }
        if (isset($_GET['iDisplayLength'])) {
            $iDisplayLength = intval($_GET['iDisplayLength']);
        }
        $sSearch = "";
        if (isset($_GET['sSearch'])) {
            $sSearch = $_GET['sSearch'];
        }

        $em = $this->getDoctrine()->getManager();
        $emisorId = null;
        $idPtoEmision = null;
        if ($this->get("security.context")->isGranted("ROLE_EMISOR_ADMIN")) {
            $emisorId = $em->getRepository('FactelBundle:User')->findEmisorId($this->get("security.context")->gettoken()->getuser()->getId());
			$idPtoEmision = $em->getRepository('FactelBundle:PtoEmision')->findIdPtoEmisionByUsuario($this->get("security.context")->gettoken()->getuser()->getId());
        } else {
			$emisorId = null;
            $idPtoEmision = $em->getRepository('FactelBundle:PtoEmision')->findIdPtoEmisionByUsuario($this->get("security.context")->gettoken()->getuser()->getId());
        }
        $count = $em->getRepository('FactelBundle:Factura')->cantidadFacturas($idPtoEmision, $emisorId);
		$entities = $em->getRepository('FactelBundle:Factura')->findFacturas($sSearch, $iDisplayStart, $iDisplayLength, $idPtoEmision, $emisorId);
        
        $totalDisplayRecords = $count;

        if ($sSearch != "") {
            $totalDisplayRecords = count($em->getRepository('FactelBundle:Factura')->findFacturas($sSearch, $iDisplayStart, 1000000, $idPtoEmision, $emisorId));
        }
        $facturaArray = array();
        $i = 0;
        foreach ($entities as $entity) {
            $fechaAutorizacion = "";
            $fechaAutorizacion = $entity->getFechaAutorizacion() != null ? $entity->getFechaAutorizacion()->format("d/m/Y H:i:s") : "";
            $facturaArray[$i] = [$entity->getId(), $entity->getEstablecimiento()->getCodigo() . "-" . $entity->getPtoEmision()->getCodigo() . "-" . $entity->getSecuencial(), $entity->getCliente()->getNombre(), $entity->getFechaEmision()->format("d/m/Y"), $fechaAutorizacion, $entity->getValorTotal(), $entity->getEstado()];
            $i++;
        }

        $arr = array(
            "iTotalRecords" => (int) $count,
            "iTotalDisplayRecords" => (int) $totalDisplayRecords,
            'aaData' => $facturaArray
        );

        $post_data = json_encode($arr);

        return new Response($post_data, 200, array('Content-Type' => 'application/json'));
    }


    /**
     * Lists all Factura entities.
     *
     * @Route("/facturas_detalle", name="all_factura_detalle")
     * @Secure(roles="ROLE_EMISOR")
     * @Method("GET")
     */
    public function facturasActionDetalle()
    {
        if (isset($_GET['sEcho'])) {
            $sEcho = $_GET['sEcho'];
        }
        $iDisplayStart = 0;
        if (isset($_GET['iDisplayStart'])) {
            $iDisplayStart = intval($_GET['iDisplayStart']);
        }
        $iDisplayLength = 100000;
        if (isset($_GET['iDisplayLength'])) {
            $iDisplayLength = intval($_GET['iDisplayLength']);
        }
        $sSearch = "";
        if (isset($_GET['sSearch'])) {
            $sSearch = $_GET['sSearch'];
        }

        $em = $this->getDoctrine()->getManager();
        $emisorId = null;
        $idPtoEmision = null;
        if ($this->get("security.context")->isGranted("ROLE_EMISOR_ADMIN")) {
            $emisorId = $em->getRepository('FactelBundle:User')->findEmisorId($this->get("security.context")->gettoken()->getuser()->getId());
        } else {
            $idPtoEmision = $em->getRepository('FactelBundle:PtoEmision')->findIdPtoEmisionByUsuario($this->get("security.context")->gettoken()->getuser()->getId());
        }
        $count = $em->getRepository('FactelBundle:Factura')->cantidadFacturas($idPtoEmision, $emisorId);
        $entities = $em->getRepository('FactelBundle:Factura')->findFacturas($sSearch, $iDisplayStart, $iDisplayLength, $idPtoEmision, $emisorId);
        $totalDisplayRecords = $count;

        if ($sSearch != "") {
            $totalDisplayRecords = count($em->getRepository('FactelBundle:Factura')->findFacturas($sSearch, $iDisplayStart, 1000000, $idPtoEmision, $emisorId));
        }
        $facturaArray = array();
        $i = 0;
        foreach ($entities as $entity) {
            $fechaAutorizacion = "";
            $fechaAutorizacion = $entity->getFechaAutorizacion() != null ? $entity->getFechaAutorizacion()->format("d/m/Y H:i:s") : "";
            $facturaArray[$i] = [$fechaAutorizacion, $entity->getEstablecimiento()->getCodigo() . "-" . $entity->getPtoEmision()->getCodigo(), $entity->getSecuencial(), $entity->getCliente()->getNombre(), $entity->getNumeroAutorizacion(), $entity->getCliente()->getIdentificacion(), $entity->getSubtotal0(), $entity->getSubtotal15(), $entity->getSubtotal5(), $entity->getIva15(), $entity->getIva5(), $entity->getValorTotal()];
            $i++;
        }

        $arr = array(
            "iTotalRecords" => (int) $count,
            "iTotalDisplayRecords" => (int) $totalDisplayRecords,
            'aaData' => $facturaArray
        );

        $post_data = json_encode($arr);

        return new Response($post_data, 200, array('Content-Type' => 'application/json'));
    }

    /**
     *
     * @Route("/cargar", name="factura_create_masivo")
     * @Method("POST")
     * @Secure(roles="ROLE_EMISOR_ADMIN")
     */
    public function createFacturaMasivaAction(Request $request)
    {
        $form = $this->createFacturaMasivaForm();
        $form->handleRequest($request);
        $em = $this->getDoctrine()->getManager();
        $user = $this->get("security.context")->gettoken()->getuser();
        $idPtoEmision = $em->getRepository('FactelBundle:PtoEmision')->findIdPtoEmisionByUsuario($user->getId());
        if (!$idPtoEmision) {
            $this->get('session')->getFlashBag()->add(
                'notice',
                "El usuario debe tener un Punto Emision asignado"
            );
            return $this->redirect($this->generateUrl('facturas_load'));
        }
        if ($form->isValid()) {

            $newFile = $form['Facturas']->getData();
            date_default_timezone_set("America/Guayaquil");
            $fecha = date("dmYHis");
            $fileName = "FacturaAutomatica-" . $fecha . ".xls";
            $newFile->move($this->getUploadRootDir(), $fileName);
            $carga = new CargaArchivo();
            $carga->setEstado("CARGADO");
            $carga->setType("FACTURA");
            $carga->setEmisor($user->getEmisor());
            $carga->setDirArchivo($fileName);
            $em->persist($carga);
            $em->flush();
        }

        return $this->redirect($this->generateUrl('facturas_load'));
    }

    /**
     *
     * @Route("/procesar-masivo-auto/{id}", name="factura_procesar_masivo_auto")
     * @Method("GET")
     * @Secure(roles="ROLE_EMISOR_ADMIN")
     */
    public function cambiarProcesarFacturaMasivaAutoAction($id)
    {
        $em = $this->getDoctrine()->getManager();
        $archivo = $em->getRepository('FactelBundle:CargaArchivo')->find($id);
        $archivo->setProcesoAutomatico(!$archivo->getProcesoAutomatico());
        if ($archivo->getEstado() == "CARGADO") {
            $em->persist($archivo);
            $em->flush();
        } else {
            $this->get('session')->getFlashBag()->add(
                'notice',
                "Solo se puede cambiar el proceso automatico de los archivos con estado CARGADO"
            );
        }
        return $this->redirect($this->generateUrl('facturas_load'));
    }

    /**
     *
     * @Route("/procesar-masivo/{id}", name="factura_procesar_masivo")
     * @Method("GET")
     * @Secure(roles="ROLE_EMISOR_ADMIN")
     */
    public function procesarFacturaMasivaAction($id)
    {
        $em = $this->getDoctrine()->getManager();
        $emisor = new \FactelBundle\Entity\Emisor();


        $archivo = $em->getRepository('FactelBundle:CargaArchivo')->find($id);
        if ($archivo) {
            if ($archivo->getEstado() == "PROCESADO") {
                $this->get('session')->getFlashBag()->add(
                    'notice',
                    "El archivo de carga de factura ya fue procesado anteriormente"
                );
                return $this->redirect($this->generateUrl('facturas_load'));
            }
            if ($archivo->getEstado() == "EN PROCESO") {
                $this->get('session')->getFlashBag()->add(
                    'notice',
                    "El archivo se encuentra actualmente en proceso"
                );
                return $this->redirect($this->generateUrl('facturas_load'));
            }
        } else {
            $this->get('session')->getFlashBag()->add(
                'notice',
                "No existe el archivo solicitado"
            );
            return $this->redirect($this->generateUrl('facturas_load'));
        }

        $ptoEmision = $em->getRepository('FactelBundle:PtoEmision')->findPtoEmisionEstabEmisorByUsuario($archivo->getCreatedBy()->getId());
        $establecimiento = $ptoEmision[0]->getEstablecimiento();
        $emisor = $establecimiento->getEmisor();

        if ($emisor != $archivo->getCreatedBy()->getEmisor()) {
            $this->get('session')->getFlashBag()->add(
                'notice',
                "Solo puede procesar los archivos cargados por su emisor"
            );
            return $this->redirect($this->generateUrl('facturas_load'));
        }

        $data = new Spreadsheet_Excel_Reader();
        $data->setOutputEncoding('UTF-8');
        $data->Spreadsheet_Excel_Reader();

        $productoCreado = 0;
        $productoActualizado = 0;

        $data->read($this->getUploadRootDir() . '/' . $archivo->getDirArchivo());
        date_default_timezone_set("America/Guayaquil");
        $archivo->setInicioProcesamiento(new \DateTime());
        $archivo->setEstado("EN PROCESO");
        $em->persist($archivo);
        $em->flush();
        $existError = false;
        for ($i = 2; $i <= $data->sheets[0]['numRows']; $i++) {
            if (isset($data->sheets[0]['cells'][$i][1]) && isset($data->sheets[0]['cells'][$i][2]) && isset($data->sheets[0]['cells'][$i][3]) && isset($data->sheets[0]['cells'][$i][4]) && isset($data->sheets[0]['cells'][$i][5]) && isset($data->sheets[0]['cells'][$i][6]) && isset($data->sheets[0]['cells'][$i][7]) && isset($data->sheets[0]['cells'][$i][8])) {
                $idFactura = "";
                try {
                    $idFactura = $data->sheets[0]['cells'][$i][1];
                    $codigoPrincipal = $data->sheets[0]['cells'][$i][3];
                    $entity = new Factura();
                    date_default_timezone_set("America/Guayaquil");
                    $fechaEmision = date("d/m/Y");
                    $entity->setEstado("CREADA");
                    $entity->setAmbiente($emisor->getAmbiente());
                    $entity->setTipoEmision($emisor->getTipoEmision());
                    $secuencial = $ptoEmision[0]->getSecuencialFactura();
                    while (strlen($secuencial) < 9) {
                        $secuencial = "0" . $secuencial;
                    }
                    $entity->setSecuencial($secuencial);
                    $entity->setClaveAcceso($this->claveAcceso($entity, $emisor, $establecimiento, $ptoEmision[0], $fechaEmision));
                    $fechaModificada = str_replace("/", "-", $fechaEmision);
                    $fecha = new \DateTime($fechaModificada);
                    $entity->setFechaEmision($fecha);
                    $identificacion = utf8_encode($data->sheets[0]['cells'][$i][7]);
                    $cliente = $em->getRepository('FactelBundle:Cliente')->findOneBy(array("identificacion" => $identificacion, "emisor" => $emisor->getId()));
                    if ($cliente == null) {
                        $cliente = new \FactelBundle\Entity\Cliente();
                        $cliente->setEmisor($emisor);
                    }

                    $cliente->setNombre(utf8_encode($data->sheets[0]['cells'][$i][8]));
                    $cliente->setTipoIdentificacion(utf8_encode($data->sheets[0]['cells'][$i][6]));
                    $cliente->setIdentificacion($identificacion);
                    if (isset($data->sheets[0]['cells'][$i][9])) {
                        $cliente->setCorreoElectronico(utf8_encode($data->sheets[0]['cells'][$i][9]));
                    }
                    $em->persist($cliente);
                    $em->flush();

                    $entity->setCliente($cliente);
                    $entity->setEmisor($emisor);
                    $entity->setEstablecimiento($establecimiento);
                    $entity->setPtoEmision($ptoEmision[0]);

                    $subTotalSinImpuesto = 0;
                    $subTotal12 = 0;
                    $subTotal0 = 0;
                    $subTotaNoObjeto = 0;
                    $subTotaExento = 0;
                    $descuento = 0;
                    $ice = 0;
                    $irbpnr = 0;
                    $iva12 = 0;
                    $propina = 0.00;
                    $valorTotal = 0;
                    $entity->setFormaPago($data->sheets[0]['cells'][$i][2]);
                    if (isset($data->sheets[0]['cells'][$i][10])) {
                        $entity->setPlazo($data->sheets[0]['cells'][$i][10]);
                    }

                    $pos = 0;
                    $productosId = array();
                    $cantidadArray = array();
                    $descuentoArray = array();
                    $error = false;
                    while (true && isset($data->sheets[0]['cells'][$i][1])) {
                        if (isset($data->sheets[0]['cells'][$i][3]) && isset($data->sheets[0]['cells'][$i][4]) && isset($data->sheets[0]['cells'][$i][5])) {
                            if ($idFactura == $data->sheets[0]['cells'][$i][1]) {
                                $codPorducto = utf8_encode($data->sheets[0]['cells'][$i][3]);
                                $productosId[$pos++] = $codPorducto;
                                $cantidadArray[$codPorducto] = $data->sheets[0]['cells'][$i][4];
                                $descuentoArray[$codPorducto] = $data->sheets[0]['cells'][$i][5];
                                $i++;
                            } else {
                                break;
                            }
                        } else {
                            $error = true;
                            break;
                        }
                    }
                    if ($error) {
                        break;
                    } else {
                        $i--;
                    }
                    $productos = array();
                    foreach ($productosId as $productoId) {
                        $producto = $em->getRepository('FactelBundle:Producto')->findBy(array("codigoPrincipal" => $productoId, "emisor" => $emisor));
                        if (count($producto) == 0) {
                            throw $this->createNotFoundException("El codigo principal " . $productoId . "  no se encuentra en el listado de productos, primeramente debe crear los productos en el sistema");
                        }
                        $productos[] = $producto[0];
                    }
                    foreach ($productos as $producto) {
                        $facturaHasProducto = new FacturaHasProducto();
                        $idProducto = $producto->getCodigoPrincipal();
                        $facturaHasProducto->setProducto($producto);
                        $impuestoIva = $producto->getImpuestoIVA();
                        $baseImponible = 0;
                        if ($impuestoIva != null) {
                            $impuesto = new Impuesto();
                            $impuesto->setCodigo("2");
                            $impuesto->setCodigoPorcentaje($impuestoIva->getCodigoPorcentaje());
                            $baseImponible = floatval($cantidadArray[$idProducto]) * floatval($producto->getPrecioUnitario()) - floatval($descuentoArray[$idProducto]);
                            $impuesto->setBaseImponible($baseImponible);

                            $impuesto->setTarifa("0");
                            $impuesto->setValor(0.00);

                            if ($impuestoIva->getCodigoPorcentaje() == "0") {
                                $subTotal0 += $baseImponible;
                            } else if ($impuestoIva->getCodigoPorcentaje() == "6") {
                                $subTotaNoObjeto += $baseImponible;
                            } else if ($impuestoIva->getCodigoPorcentaje() == "7") {
                                $subTotaExento += $baseImponible;
                            } else {
                                $impuesto->setTarifa($impuestoIva->getTarifa());
                                $impuesto->setValor(round($baseImponible * $impuestoIva->getTarifa() / 100, 2));

                                $subTotal12 += $baseImponible;
                                $iva12 += round($baseImponible * $impuestoIva->getTarifa() / 100, 2);
                            }

                            $impuesto->setFacturaHasProducto($facturaHasProducto);

                            $facturaHasProducto->addImpuesto($impuesto);
                            $subTotalSinImpuesto += $baseImponible;
                        }

                        $descuento += floatval($descuentoArray[$idProducto]);

                        $facturaHasProducto->setCantidad($cantidadArray[$idProducto]);
                        $facturaHasProducto->setPrecioUnitario($producto->getPrecioUnitario());
                        $facturaHasProducto->setDescuento($descuentoArray[$idProducto]);
                        $facturaHasProducto->setValorTotal($baseImponible);
                        $facturaHasProducto->setNombre($producto->getNombre());
                        $facturaHasProducto->setCodigoProducto($producto->getCodigoPrincipal());
                        $facturaHasProducto->setFactura($entity);
                        $entity->addFacturasHasProducto($facturaHasProducto);
                    }

                    $entity->setTotalSinImpuestos($subTotalSinImpuesto);
                    $entity->setSubtotal12($subTotal12);
                    $entity->setSubtotal0($subTotal0);
                    $entity->setSubtotalNoIVA($subTotaNoObjeto);
                    $entity->setSubtotalExentoIVA($subTotaExento);
                    $entity->setValorICE($ice);
                    $entity->setValorIRBPNR($irbpnr);
                    $entity->setIva12($iva12);
                    $entity->setTotalDescuento($descuento);
                    $entity->setPropina(0.00);
                    $importeTotal = floatval($subTotalSinImpuesto) + floatval($ice) + floatval($irbpnr) + $iva12;
                    $entity->setValorTotal($importeTotal);
                    $entity->setCargaAutomatica(false);
                    $entity->setIdFacturaCarga($idFactura);
                    $em->persist($entity);
                    $em->flush();

                    $ptoEmision[0]->setSecuencialFactura($ptoEmision[0]->getSecuencialFactura() + 1);
                    $em->persist($ptoEmision[0]);
                    $em->flush();
                } catch (\Exception $e) {
                    $error = new CargaError();
                    $error->setMessage("ID Factura: " . $idFactura . " Error: " . $e->getMessage());
                    $error->setCargaArchivo($archivo);
                    $em->persist($error);
                    $em->flush();
                    $existError = true;
                }
                $this->funtionCrearXmlPDF($entity->getId());
            }
        }
        $archivo->setfinProcesamiento(new \DateTime());
        $archivo->setEstado("PROCESADO");
        $em->persist($archivo);
        $em->flush();
        if ($existError) {
            $this->get('session')->getFlashBag()->add(
                'notice',
                "Archivo procesado con errores, favor dar click en el icono(ojo) color rojo para mas detalles"
            );
        } else {
            $this->get('session')->getFlashBag()->add(
                'confirm',
                "Archivo procesado correctamente"
            );
        }
        return $this->redirect($this->generateUrl('facturas_load'));
    }

    /**
     * Creates a new Factura entity.
     *
     * @Route("/colnar/{id}", name="factura_clonar")
     * @Method("GET")
     * @Secure(roles="ROLE_EMISOR")
     */
    public function clonarAccion($id)
    {
        $em = $this->getDoctrine()->getManager();
        $factura = new Factura();
        $entity = new Factura();
        $factura = $em->getRepository('FactelBundle:Factura')->findFacturaById($id);
        if (!$factura) {
            throw $this->createNotFoundException('No existe la factura con ID = ' + $id);
        }
        $fechaEmision = date("d/m/Y");
        $entity->setEstado("CREADA");
        $entity->setAmbiente($factura->getAmbiente());
        $entity->setTipoEmision($factura->getTipoEmision());
        $secuencial = $factura->getPtoEmision()->getSecuencialFactura();
        while (strlen($secuencial) < 9) {
            $secuencial = "0" . $secuencial;
        }
        $entity->setSecuencial($secuencial);
        $entity->setClaveAcceso($this->claveAcceso($entity, $factura->getEmisor(), $factura->getEstablecimiento(), $factura->getPtoEmision(), $fechaEmision));
        $entity->setObservacion($factura->getObservacion());
        $fechaModificada = str_replace("/", "-", $fechaEmision);
        $fecha = new \DateTime($fechaModificada);
        $entity->setFechaEmision($fecha);

        $entity->setCliente($factura->getCliente());
        $entity->setEmisor($factura->getEmisor());
        $entity->setEstablecimiento($factura->getEstablecimiento());
        $entity->setPtoEmision($factura->getPtoEmision());

        foreach ($factura->getFacturasHasProducto() as $factProducto) {
            $facturaHasProducto = new FacturaHasProducto();
            $producto = $factProducto->getProducto();
            $facturaHasProducto->setProducto($producto);
            $impuestoIva = $producto->getImpuestoIVA();
            $impuestoICE = $producto->getImpuestoICE();

            foreach ($factProducto->getImpuestos() as $factImpuesto) {
                $impuesto = new Impuesto();
                $impuesto->setCodigo($factImpuesto->getCodigo());
                $impuesto->setCodigoPorcentaje($factImpuesto->getCodigoPorcentaje());
                $impuesto->setBaseImponible($factImpuesto->getBaseImponible());
                $impuesto->setTarifa($factImpuesto->getTarifa());
                $impuesto->setValor($factImpuesto->getValor());
                $impuesto->setFacturaHasProducto($facturaHasProducto);
                $facturaHasProducto->addImpuesto($impuesto);
            }
            $facturaHasProducto->setCantidad($factProducto->getCantidad());
            $facturaHasProducto->setPrecioUnitario($factProducto->getPrecioUnitario());
            $facturaHasProducto->setDescuento($factProducto->getDescuento());
            $facturaHasProducto->setValorTotal($factProducto->getValorTotal());
            $facturaHasProducto->setNombre($factProducto->getNombre());
            $facturaHasProducto->setCodigoProducto($factProducto->getCodigoProducto());
            $facturaHasProducto->setFactura($entity);
            $entity->addFacturasHasProducto($facturaHasProducto);
        }
        $entity->setFormaPago($factura->getFormaPago());
        $entity->setPlazo($factura->getPlazo());
        $entity->setTotalSinImpuestos($factura->getTotalSinImpuestos());
        $entity->setSubtotal12($factura->getSubtotal12());
        $entity->setSubtotal0($factura->getSubtotal0());
        $entity->setSubtotalNoIVA($factura->getSubtotalNoIVA());
        $entity->setSubtotalExentoIVA($factura->getSubtotalExentoIVA());
        $entity->setValorICE($factura->getValorICE());
        $entity->setValorIRBPNR($factura->getValorIRBPNR());
        $entity->setIva12($factura->getIva12());
        $entity->setTotalDescuento($factura->getTotalDescuento());
        $entity->setPropina(0.00);
        $entity->setValorTotal($factura->getValorTotal());
        $em->persist($entity);
        $em->flush();

        $factura->getPtoEmision()->setSecuencialFactura($factura->getPtoEmision()->getSecuencialFactura() + 1);
        $em->persist($factura->getPtoEmision());
        $em->flush();

        $this->f($entity->getId());

        return $this->redirect($this->generateUrl('factura_show', array('id' => $entity->getId())));
    }

    /**
     * Creates a new Factura entity.
     *
     * @Route("/anular/{id}", name="factura_anular")
     * @Method("GET")
     * @Secure(roles="ROLE_EMISOR")
     */
    public function anularAccion($id)
    {
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('FactelBundle:Factura')->find($id);
        if (!$entity) {
            throw $this->createNotFoundException('No existe la factura con ID = ' + $id);
        }
        $entity->setEstado("ANULADA");
        $em->persist($entity);
        $em->flush();
        return $this->redirect($this->generateUrl('factura_show', array('id' => $entity->getId())));
    }

    public function funtionCrearXmlPDF($id)
    {
        $entity = new Factura();
        $procesarComprobanteElectronico = new \ProcesarComprobanteElectronico();
        $respuesta = null;
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('FactelBundle:Factura')->findFacturaById($id);
        $emisor = $entity->getEmisor();
        $configApp = new \configAplicacion();
        $configApp->dirFirma = $emisor->getDirFirma();
        $configApp->passFirma = $emisor->getPassFirma();
        $configApp->dirAutorizados = $emisor->getDirDocAutorizados();

        if ($entity->getEstablecimiento()->getDirLogo() != "") {
            $configApp->dirLogo = $entity->getEstablecimiento()->getDirLogo();
        } else {
            $configApp->dirLogo = $emisor->getDirLogo();
        }
        $configCorreo = new \configCorreo();
        $configCorreo->correoAsunto = "Nuevo Comprobante Electronico";
        $configCorreo->correoHost = $emisor->getServidorCorreo();
        $configCorreo->correoPass = $emisor->getPassCorreo();
        $configCorreo->correoPort = $emisor->getPuerto();
        $configCorreo->correoRemitente = $emisor->getCorreoRemitente();
        $configCorreo->sslHabilitado = $emisor->getSSL();


        $factura = new \factura();
        $factura->configAplicacion = $configApp;
        $factura->configCorreo = $configCorreo;

        $factura->ambiente = $entity->getAmbiente();
        $factura->tipoEmision = $entity->getTipoEmision();
        $factura->razonSocial = $emisor->getRazonSocial();
        if ($entity->getEstablecimiento()->getNombreComercial() != "") {
            $factura->nombreComercial = $entity->getEstablecimiento()->getNombreComercial();
        } else if ($emisor->getNombreComercial() != "") {
            $factura->nombreComercial = $emisor->getNombreComercial();
        }
        $factura->ruc = $emisor->getRuc(); //[Ruc]
        $factura->codDoc = "01";
        $factura->establecimiento = $entity->getEstablecimiento()->getCodigo();
        $factura->ptoEmision = $entity->getPtoEmision()->getCodigo();
        $factura->secuencial = $entity->getSecuencial();
        $factura->fechaEmision = $entity->getFechaEmision()->format("d/m/Y");
        $factura->dirMatriz = $emisor->getDireccionMatriz();

        // VERIFICAR TIPO DE RYMPE
        $getTipoEmpresa = $emisor->getTipoEmpresa();
        switch ($getTipoEmpresa) {

            case "CONTRIBUYENTE RÉGIMEN RIMPE":
                $factura->contribuyenteRimpe = "CONTRIBUYENTE RÉGIMEN RIMPE";
                break;

            case "CONTRIBUYENTE RÉGIMEN RIMPE (NEGOCIOS POPULARES)":
                $factura->contribuyenteRimpe = "CONTRIBUYENTE NEGOCIO POPULAR - RÉGIMEN RIMPE";
                break;

            case "CONTRIBUYENTE RÉGIMEN RIMPE-AGENTE RETENCION":
                $factura->agenteRetencion = 1;
                $factura->contribuyenteRimpe = "CONTRIBUYENTE RÉGIMEN RIMPE";
                break;
        }


        //$factura->agenteRetencion = "1";
        // $factura->contribuyenteRimpe ="CONTRIBUYENTE RÉGIMEN RIMPE";

        $factura->dirEstablecimiento = $entity->getEstablecimiento()->getDireccion();
        if ($emisor->getContribuyenteEspecial() != "") {
            $factura->contribuyenteEspecial = $emisor->getContribuyenteEspecial();
        }
        $factura->obligadoContabilidad = $emisor->getObligadoContabilidad();
        $factura->tipoIdentificacionComprador = $entity->getCliente()->getTipoIdentificacion();
        $factura->razonSocialComprador = $entity->getCliente()->getNombre();
        $factura->identificacionComprador = $entity->getCliente()->getIdentificacion();
        $factura->totalSinImpuestos = round($entity->getTotalSinImpuestos(),2);
        $factura->totalDescuento =  $entity->getTotalDescuento();



        $factura->propina = $entity->getPropina();
        $factura->importeTotal = $entity->getValorTotal();
        $factura->moneda = "DOLAR"; //DOLAR
        $pagos = array();

        $pago = new \pago();
        $pago->formaPago = $entity->getFormaPago();
        if ($entity->getPlazo()) {
            $pago->plazo = $entity->getPlazo();
            $pago->unidadTiempo = "Dias";
        }
        $pago->total = $entity->getValorTotal();
        $pagos[] = $pago;

        $factura->pagos = $pagos;

        $codigoPorcentajeIVA = "";
        $detalles = array();
        $facturasHasProducto = $entity->getFacturasHasProducto();
        $impuestosTotalICE = array();
        $baseImponibleICE = array();
        $impuestosTotalIRBPNR = array();
        $baseImponibleIRBPNR = array();
        foreach ($facturasHasProducto as $facturasHasProducto) {
            $producto = new \FactelBundle\Entity\Producto();
            $producto = $facturasHasProducto->getProducto();
            $detalleFactura = new \detalleFactura();
            $detalleFactura->codigoPrincipal = $facturasHasProducto->getCodigoProducto();
            if ($producto->getCodigoAuxiliar() != "") {
                $detalleFactura->codigoAuxiliar = $producto->getCodigoAuxiliar();
            }
            $detalleFactura->descripcion = $facturasHasProducto->getNombre();
            $detalleFactura->cantidad = $facturasHasProducto->getCantidad();
            $detalleFactura->precioUnitario = number_format($facturasHasProducto->getPrecioUnitario(), 2, '.', '');
            $detalleFactura->descuento = $facturasHasProducto->getDescuento();
            $detalleFactura->precioTotalSinImpuesto = round($facturasHasProducto->getValorTotal(),2);

            $impuestos = array();
            $impuestosProducto = $facturasHasProducto->getImpuestos();
            foreach ($impuestosProducto as $impuestoProducto) {

                $impuesto = new \impuesto(); // Impuesto del detalle
                $impuesto->codigo = $impuestoProducto->getCodigo();
                if ($impuestoProducto->getCodigo() == "2" && $impuestoProducto->getValor() > 0) {
                    $codigoPorcentajeIVA = $impuestoProducto->getCodigoPorcentaje();
                }
                $impuesto->codigoPorcentaje = $impuestoProducto->getCodigoPorcentaje();
                $impuesto->tarifa = $impuestoProducto->getTarifa();
                $impuesto->baseImponible = round($impuestoProducto->getBaseImponible(),2);
                $impuesto->valor = $impuestoProducto->getValor();
                $impuestos[] = $impuesto;

                if ($impuestoProducto->getCodigo() == "3") {
                    if (isset($impuestosTotalICE[$impuestoProducto->getCodigoPorcentaje()])) {
                        $impuestosTotalICE[$impuestoProducto->getCodigoPorcentaje()] += $impuestoProducto->getValor();
                        $baseImponibleICE[$impuestoProducto->getCodigoPorcentaje()] += $impuestoProducto->getBaseImponible();
                    } else {
                        $impuestosTotalICE[$impuestoProducto->getCodigoPorcentaje()] = $impuestoProducto->getValor();
                        $baseImponibleICE[$impuestoProducto->getCodigoPorcentaje()] = $impuestoProducto->getBaseImponible();
                    }
                }
                if ($impuestoProducto->getCodigo() == "5") {
                    if (isset($impuestosTotalIRBPNR[$impuestoProducto->getCodigoPorcentaje()])) {
                        $impuestosTotalIRBPNR[$impuestoProducto->getCodigoPorcentaje()] += $impuestoProducto->getValor();
                        $baseImponibleIRBPNR[$impuestoProducto->getCodigoPorcentaje()] += $impuestoProducto->getBaseImponible();
                    } else {
                        $impuestosTotalIRBPNR[$impuestoProducto->getCodigoPorcentaje()] = $impuestoProducto->getValor();
                        $baseImponibleIRBPNR[$impuestoProducto->getCodigoPorcentaje()] = $impuestoProducto->getBaseImponible();
                    }
                }
            }
            $detalleFactura->impuestos = $impuestos;
            $detalles[] = $detalleFactura;
        }

        foreach ($impuestosTotalICE as $clave => $valor) {
            $totalImpuesto = new \totalImpuesto();
            $totalImpuesto->codigo = "3";
            $totalImpuesto->codigoPorcentaje = (string) $clave;
            $totalImpuesto->baseImponible = sprintf("%01.2f", $baseImponibleICE[$clave]);
            $totalImpuesto->valor = sprintf("%01.2f", $valor);

            $totalImpuestoArray[] = $totalImpuesto;
        }

        foreach ($impuestosTotalIRBPNR as $clave => $valor) {
            $totalImpuesto = new \totalImpuesto();
            $totalImpuesto->codigo = "5";
            $totalImpuesto->codigoPorcentaje = (string) $clave;
            $totalImpuesto->baseImponible = sprintf("%01.2f", $baseImponibleIRBPNR[$clave]);
            $totalImpuesto->valor = sprintf("%01.2f", $valor);

            $totalImpuestoArray[] = $totalImpuesto;
        }

        $totalImpuestoArray = array();
        if ($entity->getSubtotal15() > 0) {
            $totalImpuesto = new \totalImpuesto();
            $totalImpuesto->codigo = "2";
            $totalImpuesto->codigoPorcentaje = "4";
            $totalImpuesto->baseImponible =  $entity->getSubtotal15();
            $totalImpuesto->valor =  $entity->getIva15();

            $totalImpuestoArray[] = $totalImpuesto;
        }
        if ($entity->getSubtotal14() > 0) {
            $totalImpuesto = new \totalImpuesto();
            $totalImpuesto->codigo = "2";
            $totalImpuesto->codigoPorcentaje = "3";
            $totalImpuesto->baseImponible = $entity->getSubtotal14();
            $totalImpuesto->valor = $entity->getIva14();

            $totalImpuestoArray[] = $totalImpuesto;
        }
        if ($entity->getSubtotal13() > 0) {
            $totalImpuesto = new \totalImpuesto();
            $totalImpuesto->codigo = "2";
            $totalImpuesto->codigoPorcentaje = "10";
            $totalImpuesto->baseImponible = $entity->getSubtotal13();
            $totalImpuesto->valor = $entity->getIva13();

            $totalImpuestoArray[] = $totalImpuesto;
        }
        if ($entity->getSubtotal12() > 0) {
            $totalImpuesto = new \totalImpuesto();
            $totalImpuesto->codigo = "2";
            $totalImpuesto->codigoPorcentaje = "2";
            $totalImpuesto->baseImponible = $entity->getSubtotal12();
            $totalImpuesto->valor = $entity->getIva12();

            $totalImpuestoArray[] = $totalImpuesto;
        }
        if ($entity->getSubtotal5() > 0) {
            $totalImpuesto = new \totalImpuesto();
            $totalImpuesto->codigo = "2";
            $totalImpuesto->codigoPorcentaje = "5";
            $totalImpuesto->baseImponible = $entity->getSubtotal5();
            $totalImpuesto->valor = $entity->getIva5();

            $totalImpuestoArray[] = $totalImpuesto;
        }
        if ($entity->getSubtotal0() > 0) {
            $totalImpuesto = new \totalImpuesto();
            $totalImpuesto->codigo = "2";
            $totalImpuesto->codigoPorcentaje = "0";
            $totalImpuesto->baseImponible = $entity->getSubtotal0();
            $totalImpuesto->valor = "0.00";

            $totalImpuestoArray[] = $totalImpuesto;
        }
        if ($entity->getSubtotalExentoIVA() > 0) {
            $totalImpuesto = new \totalImpuesto();
            $totalImpuesto->codigo = "2";
            $totalImpuesto->codigoPorcentaje = "7";
            $totalImpuesto->baseImponible = $entity->getSubtotalExentoIVA();
            $totalImpuesto->valor = "0.00";

            $totalImpuestoArray[] = $totalImpuesto;
        }
        if ($entity->getSubtotalNoIVA() > 0) {
            $totalImpuesto = new \totalImpuesto();
            $totalImpuesto->codigo = "2";
            $totalImpuesto->codigoPorcentaje = "6";
            $totalImpuesto->baseImponible = $entity->getSubtotalNoIVA();
            $totalImpuesto->valor = "0.00";

            $totalImpuestoArray[] = $totalImpuesto;
        }

        $factura->detalles = $detalles;
        $factura->totalConImpuesto = $totalImpuestoArray;

        $camposAdicionales = array();

        $cliente = $entity->getCliente();
        if ($cliente->getDireccion() != "") {
            $campoAdic = new \campoAdicional();
            $campoAdic->nombre = "Direccion";
            $campoAdic->valor = $cliente->getDireccion();

            $camposAdicionales[] = $campoAdic;
        }
        if ($cliente->getCelular() != "") {
            $campoAdic = new \campoAdicional();
            $campoAdic->nombre = "Telefono";
            $campoAdic->valor = $cliente->getCelular();

            $camposAdicionales[] = $campoAdic;
        }
        if ($cliente->getTipoIdentificacion() != "07" && $cliente->getCorreoElectronico() != "") {
            $campoAdic = new \campoAdicional();
            $campoAdic->nombre = "Email";
            $campoAdic->valor = $cliente->getCorreoElectronico();

            $camposAdicionales[] = $campoAdic;
        }
        if ($entity->getObservacion() != "") {
            $campoAdic = new \campoAdicional();
            $campoAdic->nombre = "Observacion";
            $campoAdic->valor = $entity->getObservacion();

            $camposAdicionales[] = $campoAdic;
        }
        if (count($camposAdicionales) > 0) {
            $factura->infoAdicional = $camposAdicionales;
        }

        $procesarComprobante = new \procesarComprobante();
        $procesarComprobante->comprobante = $factura;
        $procesarComprobante->envioSRI = false;
        $respuesta = $procesarComprobanteElectronico->procesarComprobante($procesarComprobante);
        if ($respuesta->return->estadoComprobante == "FIRMADO") {
            $entity->setNombreArchivo("FAC" . $entity->getEstablecimiento()->getCodigo() . "-" . $entity->getPtoEmision()->getCodigo() . "-" . $entity->getSecuencial());
        }
        $em->persist($entity);
        $em->flush();
        /*
    			$URL='https://factlite.ariasofts.com/email/tcpdf/examples/pdf_factura.php?id='.$id;
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_VERBOSE, 1);
                curl_setopt ($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
                curl_setopt ($ch, CURLOPT_URL, $URL);
                curl_setopt ($ch, CURLOPT_TIMEOUT, 120);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                $result = curl_exec ($ch);
                curl_close ($ch);
    */
    }

    /**
     * Creates a new Factura entity.
     *
     * @Route("/procesar/{id}", name="factura_procesar")
     * @Method("GET")
     * @Secure(roles="ROLE_EMISOR")
     */
    public function procesarAccion($id)
    {
        $entity = new Factura();
        $procesarComprobanteElectronico = new \ProcesarComprobanteElectronico();
        $respuesta = null;
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('FactelBundle:Factura')->findFacturaById($id);

        if (!$entity) {
            throw $this->createNotFoundException('No existe la factura con ID = ' + $id);
        }
        if ($entity->getEstado() == "AUTORIZADO") {
            $this->get('session')->getFlashBag()->add(
                'notice',
                "Este comprobante electronico ya fue autorizado"
            );
            return $this->redirect($this->generateUrl('factura_show', array('id' => $entity->getId())));
        }
        $emisor = $entity->getEmisor();
        $hoy = date("Y-m-d");
        if ($emisor->getPlan() != null && $emisor->getFechaFin()) {

            $FechaFin = $emisor->getFechaFin();
            $FechaFinString = $FechaFin->format('Y-m-d');


            if ($hoy > $FechaFinString) {
                $this->get('session')->getFlashBag()->add(
                    'notice',
                    "Su plan ha caducado por favor contacte con nuestro equipo para su renovación al siguiente número: 0979545516"
                );
                return $this->redirect($this->generateUrl('factura_show', array('id' => $entity->getId())));
            }
            if ($emisor->getCantComprobante() > $emisor->getPlan()->getCantComprobante()) {
                $this->get('session')->getFlashBag()->add(
                    'notice',
                    "Ha superado el numero de comprobantes contratado en su plan, por favor contacte con nuestro equipo para su renovación al siguiente número: 0979545516"
                );
                return $this->redirect($this->generateUrl('factura_show', array('id' => $entity->getId())));
            }
        }
		
		if($entity->getValorTotal()>50 && $entity->getCliente()->getTipoIdentificacion() == "07"){
			$this->get('session')->getFlashBag()->add(
                    'notice',
                    "La factura supera el monto máximo para Consumidor Final"
                );
                return $this->redirect($this->generateUrl('factura_show', array('id' => $entity->getId())));
		}
		
        $configApp = new \configAplicacion();
        $configApp->dirFirma = $emisor->getDirFirma();
        $configApp->passFirma = $emisor->getPassFirma();
        $configApp->dirAutorizados = $emisor->getDirDocAutorizados();

        if ($entity->getEstablecimiento()->getDirLogo() != "") {
            $configApp->dirLogo = $entity->getEstablecimiento()->getDirLogo();
        } else {
            $configApp->dirLogo = $emisor->getDirLogo();
        }
        $configCorreo = new \configCorreo();
        $configCorreo->correoAsunto = "Nuevo Comprobante Electronico";
        $configCorreo->correoHost = $emisor->getServidorCorreo();
        $configCorreo->correoPass = $emisor->getPassCorreo();
        $configCorreo->correoPort = $emisor->getPuerto();
        $configCorreo->correoRemitente = $emisor->getCorreoRemitente();
        $configCorreo->sslHabilitado = $emisor->getSSL();
        $emailCopiaOculta = null;
        if ($this->get("security.context")->gettoken()->getuser()->getCopiarEmail()) {
            $emailCopiaOculta = $this->get("security.context")->gettoken()->getuser()->getEmail();
        }
        if ($entity->getEstablecimiento()->getEmailCopia() && $entity->getEstablecimiento()->getEmailCopia() != "") {
            if ($emailCopiaOculta != "") {
                $emailCopiaOculta = $emailCopiaOculta . "," . $entity->getEstablecimiento()->getEmailCopia();
            } else {
                $emailCopiaOculta = $entity->getEstablecimiento()->getEmailCopia();
            }
        }

        if ($emailCopiaOculta) {
            $configCorreo->BBC = $emailCopiaOculta;
        }
        if ($entity->getEstado() != "PROCESANDOSE") {
            $factura = new \factura();
            $factura->configAplicacion = $configApp;
            $factura->configCorreo = $configCorreo;

            $factura->ambiente = $entity->getAmbiente();
            $factura->tipoEmision = $entity->getTipoEmision();
            $factura->razonSocial = $emisor->getRazonSocial();
            if ($entity->getEstablecimiento()->getNombreComercial() != "") {
                $factura->nombreComercial = $entity->getEstablecimiento()->getNombreComercial();
            } else if ($emisor->getNombreComercial() != "") {
                $factura->nombreComercial = $emisor->getNombreComercial();
            }
            $factura->ruc = $emisor->getRuc(); //[Ruc]
            $factura->codDoc = "01";
            $factura->establecimiento = $entity->getEstablecimiento()->getCodigo();
            $factura->ptoEmision = $entity->getPtoEmision()->getCodigo();
            $factura->secuencial = $entity->getSecuencial();
            $factura->fechaEmision = $entity->getFechaEmision()->format("d/m/Y");
            $factura->dirMatriz = $emisor->getDireccionMatriz();

            // VERIFICAR TIPO DE RYMPE
            $getTipoEmpresa = $emisor->getTipoEmpresa();
            switch ($getTipoEmpresa) {

                case "CONTRIBUYENTE RÉGIMEN RIMPE":
                    $factura->contribuyenteRimpe = "CONTRIBUYENTE RÉGIMEN RIMPE";
                    break;

                case "CONTRIBUYENTE RÉGIMEN RIMPE (NEGOCIOS POPULARES)":
                    $factura->contribuyenteRimpe = "CONTRIBUYENTE NEGOCIO POPULAR - RÉGIMEN RIMPE";
                    break;

                case "CONTRIBUYENTE RÉGIMEN RIMPE-AGENTE RETENCION":
                    $factura->agenteRetencion = 1;
                    $factura->contribuyenteRimpe = "CONTRIBUYENTE RÉGIMEN RIMPE";
                    break;
            }

            $factura->dirEstablecimiento = $entity->getEstablecimiento()->getDireccion();
            if ($emisor->getContribuyenteEspecial() != "") {
                $factura->contribuyenteEspecial = $emisor->getContribuyenteEspecial();
            }
            $factura->obligadoContabilidad = $emisor->getObligadoContabilidad();
            $factura->tipoIdentificacionComprador = $entity->getCliente()->getTipoIdentificacion();
            $factura->razonSocialComprador = $entity->getCliente()->getNombre();
            $factura->identificacionComprador = $entity->getCliente()->getIdentificacion();
            $factura->totalSinImpuestos = $entity->getTotalSinImpuestos();
            $factura->totalDescuento = $entity->getTotalDescuento();



            $factura->propina = $entity->getPropina();
            $factura->importeTotal = $entity->getValorTotal();
            $factura->moneda = "DOLAR"; //DOLAR
            $pagos = array();

            $pago = new \pago();
            $pago->formaPago = $entity->getFormaPago();
            if ($entity->getPlazo()) {
                $pago->plazo = $entity->getPlazo();
                $pago->unidadTiempo = "Dias";
            }
            $pago->total = $entity->getValorTotal();
            $pagos[] = $pago;

            $factura->pagos = $pagos;

            $codigoPorcentajeIVA = "";
            $detalles = array();
            $facturasHasProducto = $entity->getFacturasHasProducto();
            $impuestosTotalICE = array();
            $baseImponibleICE = array();
            $impuestosTotalIRBPNR = array();
            $baseImponibleIRBPNR = array();
            foreach ($facturasHasProducto as $facturasHasProducto) {
                $producto = new \FactelBundle\Entity\Producto();
                $producto = $facturasHasProducto->getProducto();
                $detalleFactura = new \detalleFactura();
                $detalleFactura->codigoPrincipal = $facturasHasProducto->getCodigoProducto();
                if ($producto->getCodigoAuxiliar() != "") {
                    $detalleFactura->codigoAuxiliar = $producto->getCodigoAuxiliar();
                }
                $detalleFactura->descripcion = $facturasHasProducto->getNombre();
                $detalleFactura->cantidad = $facturasHasProducto->getCantidad();
                $detalleFactura->precioUnitario = $facturasHasProducto->getPrecioUnitario();
                $detalleFactura->descuento = $facturasHasProducto->getDescuento();
                $detalleFactura->precioTotalSinImpuesto = $facturasHasProducto->getValorTotal();

                $impuestos = array();
                $impuestosProducto = $facturasHasProducto->getImpuestos();
                foreach ($impuestosProducto as $impuestoProducto) {

                    $impuesto = new \impuesto(); // Impuesto del detalle
                    $impuesto->codigo = $impuestoProducto->getCodigo();
                    if ($impuestoProducto->getCodigo() == "2" && $impuestoProducto->getValor() > 0) {
                        $codigoPorcentajeIVA = $impuestoProducto->getCodigoPorcentaje();
                    }
                    $impuesto->codigoPorcentaje = $impuestoProducto->getCodigoPorcentaje();
                    $impuesto->tarifa = $impuestoProducto->getTarifa();
                    $impuesto->baseImponible = $impuestoProducto->getBaseImponible();

                    $impuesto->valor = $impuestoProducto->getValor();
                    $impuestos[] = $impuesto;

                    if ($impuestoProducto->getCodigo() == "3") {
                        if (isset($impuestosTotalICE[$impuestoProducto->getCodigoPorcentaje()])) {
                            $impuestosTotalICE[$impuestoProducto->getCodigoPorcentaje()] += $impuestoProducto->getValor();
                            $baseImponibleICE[$impuestoProducto->getCodigoPorcentaje()] += $impuestoProducto->getBaseImponible();
                        } else {
                            $impuestosTotalICE[$impuestoProducto->getCodigoPorcentaje()] = $impuestoProducto->getValor();
                            $baseImponibleICE[$impuestoProducto->getCodigoPorcentaje()] = $impuestoProducto->getBaseImponible();
                        }
                    }
                    if ($impuestoProducto->getCodigo() == "5") {
                        if (isset($impuestosTotalIRBPNR[$impuestoProducto->getCodigoPorcentaje()])) {
                            $impuestosTotalIRBPNR[$impuestoProducto->getCodigoPorcentaje()] += $impuestoProducto->getValor();
                            $baseImponibleIRBPNR[$impuestoProducto->getCodigoPorcentaje()] += $impuestoProducto->getBaseImponible();
                        } else {
                            $impuestosTotalIRBPNR[$impuestoProducto->getCodigoPorcentaje()] = $impuestoProducto->getValor();
                            $baseImponibleIRBPNR[$impuestoProducto->getCodigoPorcentaje()] = $impuestoProducto->getBaseImponible();
                        }
                    }
                }
                $detalleFactura->impuestos = $impuestos;
                $detalles[] = $detalleFactura;
            }

            foreach ($impuestosTotalICE as $clave => $valor) {
                $totalImpuesto = new \totalImpuesto();
                $totalImpuesto->codigo = "3";
                $totalImpuesto->codigoPorcentaje = (string) $clave;
                $totalImpuesto->baseImponible = sprintf("%01.2f", $baseImponibleICE[$clave]);
                $totalImpuesto->valor = sprintf("%01.2f", $valor);

                $totalImpuestoArray[] = $totalImpuesto;
            }

            foreach ($impuestosTotalIRBPNR as $clave => $valor) {
                $totalImpuesto = new \totalImpuesto();
                $totalImpuesto->codigo = "5";
                $totalImpuesto->codigoPorcentaje = (string) $clave;
                $totalImpuesto->baseImponible = sprintf("%01.2f", $baseImponibleIRBPNR[$clave]);
                $totalImpuesto->valor = sprintf("%01.2f", $valor);

                $totalImpuestoArray[] = $totalImpuesto;
            }

            $totalImpuestoArray = array();
            if ($entity->getSubtotal15() > 0) {
                $totalImpuesto = new \totalImpuesto();
                $totalImpuesto->codigo = "2";
                $totalImpuesto->codigoPorcentaje = "4";
                $totalImpuesto->baseImponible = $entity->getSubtotal15();
                $totalImpuesto->valor = $entity->getIva15();

                $totalImpuestoArray[] = $totalImpuesto;
            }
            if ($entity->getSubtotal14() > 0) {
                $totalImpuesto = new \totalImpuesto();
                $totalImpuesto->codigo = "2";
                $totalImpuesto->codigoPorcentaje = "3";
                $totalImpuesto->baseImponible = $entity->getSubtotal14();
                $totalImpuesto->valor = $entity->getIva14();

                $totalImpuestoArray[] = $totalImpuesto;
            }
            if ($entity->getSubtotal13() > 0) {
                $totalImpuesto = new \totalImpuesto();
                $totalImpuesto->codigo = "2";
                $totalImpuesto->codigoPorcentaje = "10";
                $totalImpuesto->baseImponible = $entity->getSubtotal13();
                $totalImpuesto->valor = $entity->getIva13();

                $totalImpuestoArray[] = $totalImpuesto;
            }
            if ($entity->getSubtotal12() > 0) {
                $totalImpuesto = new \totalImpuesto();
                $totalImpuesto->codigo = "2";
                $totalImpuesto->codigoPorcentaje = "2";
                $totalImpuesto->baseImponible = $entity->getSubtotal12();
                $totalImpuesto->valor = $entity->getIva12();

                $totalImpuestoArray[] = $totalImpuesto;
            }
            if ($entity->getSubtotal5() > 0) {
                $totalImpuesto = new \totalImpuesto();
                $totalImpuesto->codigo = "2";
                $totalImpuesto->codigoPorcentaje = "5";
                $totalImpuesto->baseImponible = $entity->getSubtotal5();
                $totalImpuesto->valor = $entity->getIva5();

                $totalImpuestoArray[] = $totalImpuesto;
            }
            if ($entity->getSubtotal0() > 0) {
                $totalImpuesto = new \totalImpuesto();
                $totalImpuesto->codigo = "2";
                $totalImpuesto->codigoPorcentaje = "0";
                $totalImpuesto->baseImponible = $entity->getSubtotal0();
                $totalImpuesto->valor = "0.00";

                $totalImpuestoArray[] = $totalImpuesto;
            }
            if ($entity->getSubtotalExentoIVA() > 0) {
                $totalImpuesto = new \totalImpuesto();
                $totalImpuesto->codigo = "2";
                $totalImpuesto->codigoPorcentaje = "7";
                $totalImpuesto->baseImponible = $entity->getSubtotalExentoIVA();
                $totalImpuesto->valor = "0.00";

                $totalImpuestoArray[] = $totalImpuesto;
            }
            if ($entity->getSubtotalNoIVA() > 0) {
                $totalImpuesto = new \totalImpuesto();
                $totalImpuesto->codigo = "2";
                $totalImpuesto->codigoPorcentaje = "6";
                $totalImpuesto->baseImponible = $entity->getSubtotalNoIVA();
                $totalImpuesto->valor = "0.00";

                $totalImpuestoArray[] = $totalImpuesto;
            }

            $factura->detalles = $detalles;
            $factura->totalConImpuesto = $totalImpuestoArray;

            $camposAdicionales = array();

            foreach ($entity->getComposAdic() as $campoAdic) {
                $campoAdicional = new \campoAdicional();
                $campoAdicional->nombre = $campoAdic->getNombre();
                $campoAdicional->valor = $campoAdic->getValor();

                $camposAdicionales[] = $campoAdic;
            }

            $cliente = $entity->getCliente();
            if ($cliente->getDireccion() != "") {
                $campoAdic = new \campoAdicional();
                $campoAdic->nombre = "Direccion";
                $campoAdic->valor = $cliente->getDireccion();

                $camposAdicionales[] = $campoAdic;
            }
            if ($cliente->getCelular() != "") {
                $campoAdic = new \campoAdicional();
                $campoAdic->nombre = "Telefono";
                $campoAdic->valor = $cliente->getCelular();

                $camposAdicionales[] = $campoAdic;
            }

            if ($emisor->getArtesano() != "") {
                $campoAdic = new \campoAdicional();
                $campoAdic->nombre = "Artesano";
                $campoAdic->valor = "Cal. Art. N. " . $emisor->getCalificacionArtesanal();
                $camposAdicionales[] = $campoAdic;
            }

            if ($cliente->getTipoIdentificacion() != "07" && $cliente->getCorreoElectronico() != "") {
                $campoAdic = new \campoAdicional();
                $campoAdic->nombre = "Email";
                $campoAdic->valor = $cliente->getCorreoElectronico();

                $camposAdicionales[] = $campoAdic;
            }
            if ($entity->getObservacion() != "") {
                $campoAdic = new \campoAdicional();
                $campoAdic->nombre = "Observacion";
                $campoAdic->valor = $entity->getObservacion();

                $camposAdicionales[] = $campoAdic;
            }
            if (count($camposAdicionales) > 0) {
                $factura->infoAdicional = $camposAdicionales;
            }

            $procesarComprobante = new \procesarComprobante();
            $procesarComprobante->comprobante = $factura;

            if (!$entity->getFirmado() || $entity->getEstado() == "CREADA") {
                $procesarComprobante->envioSRI = false;
                $respuesta = $procesarComprobanteElectronico->procesarComprobante($procesarComprobante);
                if ($respuesta->return->estadoComprobante == "FIRMADO") {
                    $entity->setFirmado(true);
                    $procesarComprobante->envioSRI = true;
                    $respuesta = $procesarComprobanteElectronico->procesarComprobante($procesarComprobante);
                    if ($respuesta->return->estadoComprobante == "DEVUELTA" || $respuesta->return->estadoComprobante == "NO AUTORIZADO") {
                        $entity->setEnviarSiAutorizado(true);
                    }
                }
            } else if ($entity->getEstado() == "ERROR") {
                $procesarComprobante->envioSRI = true;
                $respuesta = $procesarComprobanteElectronico->procesarComprobante($procesarComprobante);
                if ($respuesta->return->estadoComprobante == "DEVUELTA" || $respuesta->return->estadoComprobante == "NO AUTORIZADO") {
                    $entity->setEnviarSiAutorizado(true);
                }
            } else if ($entity->getEnviarSiAutorizado()) {
                $procesarComprobante->envioSRI = true;
                $respuesta = $procesarComprobanteElectronico->procesarComprobante($procesarComprobante);
                if ($respuesta->return->estadoComprobante == "AUTORIZADO") {
                    $procesarComprobante->envioSRI = false;
                    $procesarComprobanteElectronico->procesarComprobante($procesarComprobante);
                }
            }
        } else {
            $comprobantePendiente = new \comprobantePendiente();

            $comprobantePendiente->configAplicacion = $configApp;
            $comprobantePendiente->configCorreo = $configCorreo;

            $comprobantePendiente->ambiente = $entity->getAmbiente();
            $comprobantePendiente->codDoc = "01";
            $comprobantePendiente->establecimiento = $entity->getEstablecimiento()->getCodigo();
            $comprobantePendiente->fechaEmision = $entity->getFechaEmision()->format("d/m/Y");
            $comprobantePendiente->ptoEmision = $entity->getPtoEmision()->getCodigo();
            $comprobantePendiente->ruc = $emisor->getRuc();
            $comprobantePendiente->secuencial = $entity->getSecuencial();
            $comprobantePendiente->tipoEmision = $entity->getTipoEmision();

            $procesarComprobantePendiente = new \procesarComprobantePendiente();
            $procesarComprobantePendiente->comprobantePendiente = $comprobantePendiente;

            $respuesta = $procesarComprobanteElectronico->procesarComprobantePendiente($procesarComprobantePendiente);
            if ($respuesta->return->estadoComprobante == "PROCESANDOSE") {
                $respuesta->return->estadoComprobante = "ERROR";
            }
        }


        if ($respuesta->return->mensajes != null) {
            $mensajesArray = array();
            if (is_array($respuesta->return->mensajes)) {
                $mensajesArray = $respuesta->return->mensajes;
            } else {
                $mensajesArray[] = $respuesta->return->mensajes;
            }
            foreach ($mensajesArray as $mensaje) {
                if ($mensaje->identificador == "43") {
                    $comprobantePendiente = new \comprobantePendiente();

                    $comprobantePendiente->configAplicacion = $configApp;
                    $comprobantePendiente->configCorreo = $configCorreo;

                    $comprobantePendiente->ambiente = $entity->getAmbiente();
                    $comprobantePendiente->codDoc = "01";
                    $comprobantePendiente->establecimiento = $entity->getEstablecimiento()->getCodigo();
                    $comprobantePendiente->fechaEmision = $entity->getFechaEmision()->format("d/m/Y");
                    $comprobantePendiente->ptoEmision = $entity->getPtoEmision()->getCodigo();
                    $comprobantePendiente->ruc = $emisor->getRuc();
                    $comprobantePendiente->secuencial = $entity->getSecuencial();
                    $comprobantePendiente->tipoEmision = $entity->getTipoEmision();

                    $procesarComprobantePendiente = new \procesarComprobantePendiente();
                    $procesarComprobantePendiente->comprobantePendiente = $comprobantePendiente;

                    $respuesta = $procesarComprobanteElectronico->procesarComprobantePendiente($procesarComprobantePendiente);

                    break;
                }
            }
        }
        $entity->setNumeroAutorizacion($respuesta->return->numeroAutorizacion);

        if ($respuesta->return->fechaAutorizacion != "") {
            $fechaAutorizacion = str_replace("/", "-", $respuesta->return->fechaAutorizacion);
            $entity->setFechaAutorizacion(new \DateTime($fechaAutorizacion));
        }
        $entity->setEstado($respuesta->return->estadoComprobante);
        if ($entity->getEstado() == "AUTORIZADO") {
            $entity->setNombreArchivo("FAC" . $entity->getEstablecimiento()->getCodigo() . "-" . $entity->getPtoEmision()->getCodigo() . "-" . $entity->getSecuencial());
            if ($emisor->getAmbiente() == "2") {
                $emisor->setCantComprobante($emisor->getCantComprobante() + 1);
                $em->persist($emisor);
            }

            $em = $this->getDoctrine()->getManager();
			
            $factura = new Factura();
            $factura = $em->getRepository('FactelBundle:Factura')->findFacturaById($id);
            $archivoName = $factura->getNombreArchivo();
            $claveAcceso = $factura->getClaveAcceso();
            $claveAcceso2 = $factura->getNumeroAutorizacion();
            $total_factura = $factura->getValorTotal();
            $pathPDF = $factura->getEmisor()->getDirDocAutorizados() . DIRECTORY_SEPARATOR . $factura->getCliente()->getIdentificacion() . DIRECTORY_SEPARATOR . $archivoName . ".pdf";


            if (file_exists($pathPDF)) {
                copy($pathPDF, '/var/www/html/web/upload/pdf/' . $claveAcceso2 . '.pdf');
            }


            if (strlen($entity->getCliente()->getCelular()) == 10) {
                $celular = $entity->getCliente()->getCelular();
                $rest = substr($celular, 0, 2);
                if ($rest == '09') {
                    if ($entity->getCliente()->getIdentificacion() == '9999999999999') {
                        $telefono = '593989864904';
                    } else {
                        $telefono = '593' . substr($entity->getCliente()->getCelular(), 1, strlen($entity->getCliente()->getCelular()));
                    }
			
			
                    sleep(3);
                    $send = '{
                        "message": "*Estimad@, ' . $entity->getCliente()->getNombre() . '* \nTe notificamos que el siguiente Documento Electrónico ha sido generado con éxito \r\n\r\n *FACTURA No.:*           \r\n ' . $entity->getEstablecimiento()->getCodigo() . "-" . $entity->getPtoEmision()->getCodigo() . "-" . $entity->getSecuencial() . '  \r\n\r\n *CLAVE DE ACCESO :* \r\n' . $claveAcceso2 . ' \r\n\r\n *Fecha de Emisión :* \r\n ' . $respuesta->return->fechaAutorizacion . ' \r\n\r\n *Valor Total:* ' . $total_factura . ' 💵  \r\n\r\nGenerado automáticamente por FacturaLite. Visítanos\n www.facturalite.plus  \r\n\r\n"
                        }';
			
                    // Bryan
                    $clienteNombre = $factura->getCliente()->getNombre();		
					$emisor = $entity->getEmisor();
					$empresaRazonSocial = $emisor->getRazonSocial();					
                    $secuencia = $entity->getEstablecimiento()->getCodigo() . "-" . $entity->getPtoEmision()->getCodigo() . "-" . $entity->getSecuencial();
                    $fechaR = $respuesta->return->fechaAutorizacion;
			
                    $message = "*Estimad@,  $clienteNombre* \nTe notificamos que el siguiente Documento Electrónico ha sido generado con éxito \r\n*POR: $empresaRazonSocial* \r\n\r\n *FACTURA No.:*           \r\n  $secuencia   \r\n\r\n *CLAVE DE ACCESO :* \r\n $claveAcceso2  \r\n\r\n *Fecha de Emisión :* \r\n  $fechaR  \r\n\r\n *Valor Total:*  $total_factura  💵  \r\n\r\nGenerado automáticamente por FacturaLite. Visítanos\n www.facturalite.plus  \r\n\r\n";
			
					//$chat = [
                    //    "secret" => "2c7f0226e318534decab6c899cb5dd433083fdb5", // reemplaza con tu API secret
                    //    "account" => "1733770132c81e728d9d4c2f636f067f89cc14862c67573b94b269a", // reemplaza con tu account ID
                    //    "recipient" => "+$telefono", // número de teléfono del destinatario
                    //    "type" => "document",
                    //    "document_url" => "https://factlite.ariasofts.com/upload/pdf/$claveAcceso2.pdf",
                    //    "document_type" => "pdf",
                    //    "document_name" => "$claveAcceso2.pdf",
                    //    "message" => $message // mensaje que quieres enviar
                    //];
					//
                    //// Inicializa cURL
                    //$cURL = curl_init("https://sender.codigo.plus/api/send/whatsapp");
					//
                    //// Configura las opciones de cURL
                    //curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    //curl_setopt($cURL, CURLOPT_POSTFIELDS, http_build_query($chat)); // utiliza http_build_query para convertir el array en un query string
					//
                    //// Ejecuta cURL y cierra la sesión
                    //$response = curl_exec($cURL);
                    //curl_close($cURL);
					
					$payload = [
						"number" => $telefono,
						"body" => $message,
						"userId" => 442, // ← reemplaza con el tuyo
						"queueId" => 293, // ← reemplaza con el tuyo
						"sendSignature" => true,
						"closeTicket" => false,
						"document_url" => "https://factlite.ariasofts.com/upload/pdf/$claveAcceso2.pdf",
                        "document_type" => "pdf",
                        "document_name" => "$claveAcceso2.pdf"
					];
			
					// Inicializa cURL para Chateam
					$ch = curl_init("https://server.chateam.ws/api/messages/send");
			
					curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
					curl_setopt($ch, CURLOPT_POST, true);
					curl_setopt($ch, CURLOPT_HTTPHEADER, [
						"Content-Type: application/json",
						"Authorization: Bearer aIT1Kyz7HV8DUWkOPzKsA70wboKRIW" // ← reemplaza con tu token real
					]);
					curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
			
					$response = curl_exec($ch);
					$http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
					curl_close($ch);
                }
            }	
        }
        $mensajes = $entity->getMensajes();
        foreach ($mensajes as $mensaje) {
            $em->remove($mensaje);
        }
        if ($respuesta->return->mensajes != null) {
            $mensajesArray = array();
            if (is_array($respuesta->return->mensajes)) {
                $mensajesArray = $respuesta->return->mensajes;
            } else {
                $mensajesArray[] = $respuesta->return->mensajes;
            }
            foreach ($mensajesArray as $mensaje) {
                $mensajeGenerado = new \FactelBundle\Entity\Mensaje();
                $mensajeGenerado->setIdentificador($mensaje->identificador);
                $mensajeGenerado->setMensaje($mensaje->mensaje);
                $mensajeGenerado->setInformacionAdicional($mensaje->informacionAdicional);
                $mensajeGenerado->setTipo($mensaje->tipo);
                $mensajeGenerado->setFactura($entity);
                $em->persist($mensajeGenerado);
            }
        }
        $em->persist($entity);
        $em->flush();

        return $this->redirect($this->generateUrl('factura_show', array('id' => $entity->getId())));
    }

    /**
     * Creates a new Factura entity.
     *
     * @Route("/enviarEmail/{id}", name="factura_enviar_email")
     * @Method("POST")
     * @Secure(roles="ROLE_EMISOR")
     */

    public function sendEmail(Request $request, $id)
    {


        $destinatario = $request->request->get("email");
        $emails = explode(',', $destinatario); 
		$correoSuccess = true;
		$whatsappSuccess = false;
		$correoError = "";
		$whatsappError = "";


        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('FactelBundle:Factura')->findFacturaById($id);
        // $URL='https://factlite.ariasofts.com/email/send_correo.php?correo='.$destinatario.'&id='.$id;
        // $ch = curl_init();
        // curl_setopt($ch, CURLOPT_VERBOSE, 1);
        // curl_setopt ($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
        // curl_setopt ($ch, CURLOPT_URL, $URL);
        // curl_setopt ($ch, CURLOPT_TIMEOUT, 120);
        // curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        // $result = curl_exec ($ch);
        // curl_close ($ch);

        foreach ($emails as $destinatario) {
            $destinatario = trim($destinatario); // Elimina espacios en blanco alrededor de la dirección de correo
            $URL = 'https://factlite.ariasofts.com/email/send_correo.php?correo=' . urlencode($destinatario) . '&id=' . $id;

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_VERBOSE, 1);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
            curl_setopt($ch, CURLOPT_URL, $URL);
            curl_setopt($ch, CURLOPT_TIMEOUT, 120);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            $result = curl_exec($ch);
            curl_close($ch);
			
			$jsonArray = json_decode($result, true);
			if (!isset($jsonArray['result']) || $jsonArray['result'] != 1) {
				$correoSuccess = false;
				$correoError = $jsonArray['error'];
			}
        }

        $factura = new Factura();
        $factura = $em->getRepository('FactelBundle:Factura')->findFacturaById($id);
        $archivoName = $factura->getNombreArchivo();
        $claveAcceso = $factura->getClaveAcceso();
        $claveAcceso2 = $factura->getNumeroAutorizacion();
        $total_factura = $factura->getValorTotal();
		
        $pathPDF = $factura->getEmisor()->getDirDocAutorizados() . DIRECTORY_SEPARATOR . $factura->getCliente()->getIdentificacion() . DIRECTORY_SEPARATOR . $archivoName . ".pdf";

        if (strlen($entity->getCliente()->getCelular()) == 10) {
            $celular = $entity->getCliente()->getCelular();
            $rest = substr($celular, 0, 2);
            if ($rest == '09') {
                if ($entity->getCliente()->getIdentificacion() == '9999999999999') {
                    $telefono = '593989864904';
                } else {
                    $telefono = '593' . substr($entity->getCliente()->getCelular(), 1, strlen($entity->getCliente()->getCelular()));
                }

                sleep(3);

                $clienteNombre = $factura->getCliente()->getNombre();
                $secuencia = $entity->getEstablecimiento()->getCodigo() . "-" . $entity->getPtoEmision()->getCodigo() . "-" . $entity->getSecuencial();
                $fechaR = $factura->getFechaAutorizacion();

                if ($fechaR instanceof \DateTime) {
                    $fechaRFormatted = $fechaR->format('Y-m-d H:i:s'); // Ajusta el formato según tus necesidades
                }
				$emisor = $entity->getEmisor();
				$empresaRazonSocial = $emisor->getRazonSocial();	

                $message = "*Estimad@,  $clienteNombre* \nTe notificamos que el siguiente Documento Electrónico ha sido generado con éxito \r\n*POR: $empresaRazonSocial* \r\n\r\n *FACTURA No.:*        \r\n  $secuencia   \r\n\r\n *CLAVE DE ACCESO :* \r\n $claveAcceso2  \r\n\r\n *Fecha de Emisión :* \r\n  $fechaRFormatted  \r\n\r\n *Valor Total:*  $total_factura  💵  \r\n\r\nGenerado automáticamente por FacturaLite. Visítanos\n www.facturalite.plus  \r\n\r\n";

            		//$chat = [
                    //    "secret" => "2c7f0226e318534decab6c899cb5dd433083fdb5", // reemplaza con tu API secret
                    //    "account" => "1733770132c81e728d9d4c2f636f067f89cc14862c67573b94b269a", // reemplaza con tu account ID
                    //    "recipient" => "+$telefono", // número de teléfono del destinatario
                    //    "type" => "document",
                    //    "document_url" => "https://factlite.ariasofts.com/upload/pdf/$claveAcceso2.pdf",
                    //    "document_type" => "pdf",
                    //    "document_name" => "$claveAcceso2.pdf",
                    //    "message" => $message // mensaje que quieres enviar
                    //];
					//
                    //// Inicializa cURL
                    //$cURL = curl_init("https://sender.codigo.plus/api/send/whatsapp");
					//
                    //// Configura las opciones de cURL
                    //curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    //curl_setopt($cURL, CURLOPT_POSTFIELDS, http_build_query($chat)); // utiliza http_build_query para convertir el array en un query string
					//
                    //// Ejecuta cURL y cierra la sesión
                    //$response = curl_exec($cURL);
                    //curl_close($cURL);
					
					$payload = [
						"number" => $telefono,
						"body" => $message,
						"userId" => 442, // ← reemplaza con el tuyo
						"queueId" => 293, // ← reemplaza con el tuyo
						"sendSignature" => true,
						"closeTicket" => false,
						"document_url" => "https://factlite.ariasofts.com/upload/pdf/$claveAcceso2.pdf",
                        "document_type" => "pdf",
                        "document_name" => "$claveAcceso2.pdf"
					];

					// Inicializa cURL para Chateam
					$ch = curl_init("https://server.chateam.ws/api/messages/send");

					curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
					curl_setopt($ch, CURLOPT_POST, true);
					curl_setopt($ch, CURLOPT_HTTPHEADER, [
						"Content-Type: application/json",
						"Authorization: Bearer aIT1Kyz7HV8DUWkOPzKsA70wboKRIW" // ← reemplaza con tu token real
					]);
					curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));

					$response = curl_exec($ch);
					$http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
					curl_close($ch);
					
					$jsonArray2 = json_decode($response, true);
					if ($http_status == 200 && isset($jsonArray2['result']) && $jsonArray2['result'] == 200) {
						$whatsappSuccess = true;
					} else {
						$whatsappError = "Error al enviar por WhatsApp.";
					}
            }
        }

		if ($correoSuccess && $whatsappSuccess) {
			$this->get('session')->getFlashBag()->add('confirm', "Correo y WhatsApp enviados con éxito.");
		} elseif ($correoSuccess && !$whatsappSuccess) {
			$this->get('session')->getFlashBag()->add('notice', "Correo enviado correctamente, pero error en WhatsApp: $whatsappError");
		} elseif (!$correoSuccess && $whatsappSuccess) {
			$this->get('session')->getFlashBag()->add('notice', "WhatsApp enviado correctamente, pero error en correo: $correoError");
		} else {
			$this->get('session')->getFlashBag()->add('error', "No se pudo enviar ni el correo ni el WhatsApp.\nCorreo: $correoError\nWhatsApp: $whatsappError");
		}
        //$jsonArray = json_decode($result, true);
		//$jsonArray2 = json_decode($response, true);
        //if ($jsonArray['result'] == 1 && $jsonArray2['result'] == 200) {
        //    $this->get('session')->getFlashBag()->add(
        //        'confirm',
        //        "Correo y whatsapp enviados con éxito"
        //    );
        //} else {
        //    $this->get('session')->getFlashBag()->add(
        //        'notice',
        //        $jsonArray['error']
        //    );
        //}
        return $this->redirect($this->generateUrl('factura_show', array('id' => $entity->getId())));
    }




    /**
     * Creates a new Factura entity.
     *
     * @Route("/", name="factura_create")
     * @Method("POST")
     * @Secure(roles="ROLE_EMISOR")
     * @Template("FactelBundle:Factura:new.html.twig")
     */
    public function createAction(Request $request)
    {
        $secuencial = $request->request->get("secuencial");
        $fechaEmision = $request->request->get("fechaEmision");
        $idCliente = $request->request->get("idCliente");
        $nombre = $request->request->get("nombre");
        $celular = $request->request->get("celular");
        $email = $request->request->get("email");
        $tipoIdentificacion = $request->request->get("tipoIdentificacion");
        $identificacion = $request->request->get("identificacion");
        $direccion = $request->request->get("direccion");
        $nuevoCliente = $request->request->get("nuevoCliente");
        $idFactura = $request->request->get("idFactura");
        $formaPago = $request->request->get("formaPago");
        $plazo = $request->request->get("plazo");
        $observacion = $request->request->get("observacion");

        $texto = "";
        $campos = "";
        $cantidadErrores = 0;
        if ($secuencial == '') {
            $campos .= "Secuencial, ";
            $cantidadErrores++;
        }
        if ($fechaEmision == '') {
            $campos .= "Fecha Emision, ";
            $cantidadErrores++;
        }
        if ($nombre == '') {
            $campos .= "Nombre Cliente, ";
            $cantidadErrores++;
        }
        if ($tipoIdentificacion == '') {
            $campos .= "Tipo Identificacion, ";
            $cantidadErrores++;
        }
        if ($identificacion == '') {
            $campos .= "Identificacion, ";
            $cantidadErrores++;
        }
        if ($formaPago == '') {
            $campos .= "Forma Pago, ";
            $cantidadErrores++;
        }
        if ($cantidadErrores > 0) {
            if ($cantidadErrores == 1) {
                $texto = "El campo <strong>" . $campos . "</strong> no puede estar vacios";
            } else {
                $texto = "Los campos " . $campos . " no pueden estar vacios";
            }
            $this->get('session')->getFlashBag()->add(
                'notice',
                $texto
            );

            return $this->redirect($this->generateUrl('factura_new', array()));
        }
        $em = $this->getDoctrine()->getManager();
        if ($idFactura != null && $idFactura != '') {
            $entity = new Factura();
            $entity = $em->getRepository('FactelBundle:Factura')->find($idFactura);
            if (!is_null($entity)) {
                $mensajes = $entity->getMensajes();
                foreach ($mensajes as $mensaje) {
                    $em->remove($mensaje);
                }
                $facturasHasProducto = $entity->getFacturasHasProducto();
                foreach ($facturasHasProducto as $facturaHasProducto) {
                    foreach ($facturaHasProducto->getImpuestos() as $impuesto) {
                        $em->remove($impuesto);
                    }
                    $em->remove($facturaHasProducto);
                }
                $em->flush();
            }
        } else {
            $entity = new Factura();
        }



        $ptoEmision = $em->getRepository('FactelBundle:PtoEmision')->findPtoEmisionEstabEmisorByUsuario($this->get("security.context")->gettoken()->getuser()->getId());

        if ($ptoEmision != null && count($ptoEmision) > 0) {
            $establecimiento = $ptoEmision[0]->getEstablecimiento();
            $emisor = $establecimiento->getEmisor();

            $entity->setEstado("CREADA");
            $entity->setAmbiente($emisor->getAmbiente());
            $entity->setTipoEmision($emisor->getTipoEmision());
			
			//$pto = $ptoEmision[0]; // Objeto PtoEmision
			//
			//$secuencial = $pto->getSecuencialFactura();
			//while (strlen($secuencial) < 9) {
			//	$secuencial = "0" . $secuencial;
			//}
			
			$entity->setSecuencial($secuencial);
			
            $entity->setClaveAcceso($this->claveAcceso($entity, $emisor, $establecimiento, $ptoEmision[0], $fechaEmision));
            $entity->setObservacion($observacion);
            $fechaModificada = str_replace("/", "-", $fechaEmision);
            $fecha = new \DateTime($fechaModificada);
            $entity->setFechaEmision($fecha);
            $cliente = $em->getRepository('FactelBundle:Cliente')->find($idCliente);
            if ($nuevoCliente) {
                $emisorId = $this->get("security.context")->gettoken()->getuser()->getEmisor()->getId();
                if ($em->getRepository('FactelBundle:Cliente')->findBy(array("identificacion" => $identificacion, "emisor" => $emisorId)) != null) {
                    $this->get('session')->getFlashBag()->add(
                        'notice',
                        "La identificación del sujeto retenido ya se encuentra resgistrada. Utilice la opción de búsqueda"
                    );
                    return $this->redirect($this->generateUrl('retencion_new', array()));
                }
                $cliente = new \FactelBundle\Entity\Cliente();
                $emisor = $em->getRepository('FactelBundle:Emisor')->find($emisorId);
                $cliente->setEmisor($emisor);
            }

            $cliente->setNombre($nombre);
            $cliente->setTipoIdentificacion($tipoIdentificacion);
            $cliente->setIdentificacion($identificacion);
            $cliente->setCelular($celular);
            $cliente->setCorreoElectronico($email);
            $cliente->setDireccion($direccion);
            $em->persist($cliente);
            $em->flush();


            $entity->setCliente($cliente);
            $entity->setEmisor($emisor);
            $entity->setEstablecimiento($establecimiento);
            $entity->setPtoEmision($ptoEmision[0]);

            $subTotalSinImpuesto = 0;
            $subTotal15 = 0;
            $subTotal14 = 0;
            $subTotal13 = 0;
            $subTotal12 = 0;
            $subTotal5 = 0;
            $subTotal0 = 0;
            $subTotaNoObjeto = 0;
            $subTotaExento = 0;
            $descuento = 0;
            $ice = 0;
            $irbpnr = 0;
            $iva15 = 0;
            $iva14 = 0;
            $iva13 = 0;
            $iva12 = 0;
            $iva5 = 0;
            $propina = 0;
            $valorTotal = 0;

            $idProductoArray = $request->request->get("idProducto");
            if ($idProductoArray == null) {
                $this->get('session')->getFlashBag()->add(
                    'notice',
                    "La factura debe contener al menos un producto"
                );
                return $this->redirect($this->generateUrl('factura_new', array()));
            }
            $productos = $em->getRepository('FactelBundle:Producto')->findById($idProductoArray);
            if (count($productos) == 0) {
                $this->get('session')->getFlashBag()->add(
                    'notice',
                    "Los productos solicitados para esta factura no se encuentran disponibles"
                );
                return $this->redirect($this->generateUrl('factura_new', array()));
            }
            foreach ($productos as $producto) {
                $facturaHasProducto = new FacturaHasProducto();
                $idProducto = $producto->getId();

                $facturaHasProducto->setProducto($producto);
                $impuestoIva = $producto->getImpuestoIVA();
                $impuestoICE = $producto->getImpuestoICE();
                $impuestoIRBPNR = $producto->getImpuestoIRBPNR();

                $cantidadArray = $request->request->get("cantidad");
                $descuentoArray = $request->request->get("descuento");
                $precioUnitario = $request->request->get("precio");
                $nombreProducto = $request->request->get("nombreProducto");
                $codigoProducto = $request->request->get("codigoProducto");
                $iceArray = $request->request->get("ice");
                $irbpnrArray = $request->request->get("irbpnr");
                $baseImponible = 0;



                if ($impuestoIva != null) {
                    $impuesto = new Impuesto();
                    $impuesto->setCodigo("2");
                    $impuesto->setCodigoPorcentaje($impuestoIva->getCodigoPorcentaje());
                    $baseImponible = round(floatval($cantidadArray[$idProducto]) * floatval($precioUnitario[$idProducto]) - floatval($descuentoArray[$idProducto]),2);
                    $impuesto->setBaseImponible($baseImponible);


                    $impuesto->setTarifa("0");
                    $impuesto->setValor(0.00);

                    if ($impuestoIva->getCodigoPorcentaje() == "0") {
                        $subTotal0 += $baseImponible;
                    } else if ($impuestoIva->getCodigoPorcentaje() == "6") {
                        $subTotaNoObjeto += $baseImponible;
                    } else if ($impuestoIva->getCodigoPorcentaje() == "7") {
                        $subTotaExento += $baseImponible;
                    }
                    // IVA 15
                    else if ($impuestoIva->getCodigoPorcentaje() == "4") {
                        $impuesto->setTarifa($impuestoIva->getTarifa());
                        $impuesto->setValor(round($baseImponible * $impuestoIva->getTarifa() / 100, 2));

                        $subTotal15 += $baseImponible;
                        $iva15 += round($baseImponible * $impuestoIva->getTarifa() / 100, 2);
                    }
                    // IVA 14
                    else if ($impuestoIva->getCodigoPorcentaje() == "3") {
                        $impuesto->setTarifa($impuestoIva->getTarifa());
                        $impuesto->setValor(round($baseImponible * $impuestoIva->getTarifa() / 100, 2));

                        $subTotal14 += $baseImponible;
                        $iva14 += round($baseImponible * $impuestoIva->getTarifa() / 100, 2);
                    }
                    // IVA 13
                    else if ($impuestoIva->getCodigoPorcentaje() == "10") {
                        $impuesto->setTarifa($impuestoIva->getTarifa());
                        $impuesto->setValor(round($baseImponible * $impuestoIva->getTarifa() / 100, 2));

                        $subTotal13 += $baseImponible;
                        $iva13 += round($baseImponible * $impuestoIva->getTarifa() / 100, 2);
                    }
                    // IVA 5
                    else if ($impuestoIva->getCodigoPorcentaje() == "5") {
                        $impuesto->setTarifa($impuestoIva->getTarifa());
                        $impuesto->setValor(round($baseImponible * $impuestoIva->getTarifa() / 100, 2));

                        $subTotal5 += $baseImponible;
                        $iva5 += round($baseImponible * $impuestoIva->getTarifa() / 100, 2);
                    }
                    // IVA 12
                    else {
                        $impuesto->setTarifa($impuestoIva->getTarifa());
                        $impuesto->setValor(round($baseImponible * $impuestoIva->getTarifa() / 100, 2));

                        $subTotal12 += $baseImponible;
                        $iva12 += round($baseImponible * $impuestoIva->getTarifa() / 100, 2);
                    }

                    $impuesto->setFacturaHasProducto($facturaHasProducto);

                    $facturaHasProducto->addImpuesto($impuesto);
                    $subTotalSinImpuesto += $baseImponible;
                }
                if ($impuestoICE != null) {
                    $impuesto = new Impuesto();
                    $impuesto->setCodigo("3");
                    $impuesto->setCodigoPorcentaje($impuestoICE->getCodigoPorcentaje());
                    $impuesto->setTarifa("0");
                    $impuesto->setBaseImponible($baseImponible);
                    $impuesto->setValor($iceArray[$idProducto]);

                    $impuesto->setFacturaHasProducto($facturaHasProducto);

                    $facturaHasProducto->addImpuesto($impuesto);
                    $ice += floatval($iceArray[$idProducto]);
                }

                if ($impuestoIRBPNR != null) {
                    $impuesto = new Impuesto();
                    $impuesto->setCodigo("5");
                    $impuesto->setCodigoPorcentaje($impuestoIRBPNR->getCodigoPorcentaje());
                    $impuesto->setTarifa("0");
                    $impuesto->setBaseImponible($baseImponible);
                    $impuesto->setValor($irbpnrArray[$idProducto]);

                    $impuesto->setFacturaHasProducto($facturaHasProducto);

                    $facturaHasProducto->addImpuesto($impuesto);
                    $irbpnr += floatval($irbpnrArray[$idProducto]);
                }
                $descuento += floatval($descuentoArray[$idProducto]);

                $facturaHasProducto->setCantidad($cantidadArray[$idProducto]);
                $facturaHasProducto->setPrecioUnitario($precioUnitario[$idProducto]);
                $facturaHasProducto->setDescuento($descuentoArray[$idProducto]);
                $facturaHasProducto->setValorTotal($baseImponible);
                $facturaHasProducto->setNombre($nombreProducto[$idProducto]);
                $facturaHasProducto->setCodigoProducto($codigoProducto[$idProducto]);
                $facturaHasProducto->setFactura($entity);
                $entity->addFacturasHasProducto($facturaHasProducto);
            }
            $entity->setFormaPago($formaPago);
            if ($plazo) {
                $entity->setPlazo($plazo);
            }


            $entity->setTotalSinImpuestos($subTotalSinImpuesto);
            $entity->setSubtotal15(round($subTotal15,2));
            $entity->setSubtotal14(round($subTotal14,2));
            $entity->setSubtotal13(round($subTotal13,2));
            $entity->setSubtotal12(round($subTotal12,2));
            $entity->setSubtotal5 (round($subTotal5,2));
            $entity->setSubtotal0 (round($subTotal0,2));
            $entity->setSubtotalNoIVA(round($subTotaNoObjeto,2));
            $entity->setSubtotalExentoIVA(round($subTotaExento,2));
            $entity->setValorICE($ice);
            $entity->setValorIRBPNR($irbpnr);
            $entity->setIva15($iva15);
            $entity->setIva14($iva14);
            $entity->setIva13($iva13);
            $entity->setIva12($iva12);
            $entity->setIva5($iva5);
            $entity->setTotalDescuento($descuento);
            $entity->setPropina(0);
            $importeTotal = round((floatval($subTotalSinImpuesto) + floatval($ice) + floatval($irbpnr) + $iva15 + $iva14 + $iva13 + $iva12 + $iva5),2);

            $entity->setValorTotal($importeTotal);
            $em->persist($entity);
            $em->flush();
            if ($idFactura == null || $idFactura == '') {
                $ptoEmision[0]->setSecuencialFactura($ptoEmision[0]->getSecuencialFactura() + 1);
                $em->persist($ptoEmision[0]);
                $em->flush();
            }
            $this->funtionCrearXmlPDF($entity->getId());
			if($request->request->get("automatico") == "true"){
				return $this->procesarAccion($entity->getId());
			}else{
				return $this->redirect($this->generateUrl('factura_show',array('id' => $entity->getId())));
			}
           
        } else {
            throw $this->createNotFoundException('El usuario del sistema no tiene asignado un Punto de Emision.');
        }
    }


    /**
     * Creates a new Factura entity.
     *
     * @Route("/descargar-small/{id}", name="factura_descargar_small")
     * @Method("GET")
     * @Secure(roles="ROLE_EMISOR")
     */
    public function descargarSmallAction($id)
    {
        $em = $this->getDoctrine()->getManager();
        $factura = new Factura();
        $factura = $em->getRepository('FactelBundle:Factura')->findFacturaById($id);
        $archivoName = $factura->getNombreArchivo();
        $pathPDF = $factura->getEmisor()->getDirDocAutorizados() . DIRECTORY_SEPARATOR . $factura->getCliente()->getIdentificacion() . DIRECTORY_SEPARATOR . $archivoName . "-small.pdf";

        $response = new Response();
        //then send the headers to foce download the zip file
        $response->headers->set('Content-Type', 'application/pdf');
        $response->headers->set('Content-Disposition', 'attachment; filename="' . basename($pathPDF) . '"');
        $response->headers->set('Pragma', "no-cache");
        $response->headers->set('Expires', "0");
        $response->headers->set('Content-Transfer-Encoding', "binary");
        $response->sendHeaders();
        $response->setContent(readfile($pathPDF));
        return $response;
    }

    /**
     * Creates a new Factura entity.
     *
     * @Route("/descargar/{id}/{type}", name="factura_descargar")
     * @Method("GET")
     * @Secure(roles="ROLE_EMISOR")
     */
    public function descargarAction($id, $type = "zip")
    {
        $em = $this->getDoctrine()->getManager();
        $factura = new Factura();
        $factura = $em->getRepository('FactelBundle:Factura')->findFacturaById($id);

        $archivoName = $factura->getNombreArchivo();
        $pathXML = $factura->getEmisor()->getDirDocAutorizados() . DIRECTORY_SEPARATOR . $factura->getCliente()->getIdentificacion() . DIRECTORY_SEPARATOR . $archivoName . ".xml";
        $pathPDF = $factura->getEmisor()->getDirDocAutorizados() . DIRECTORY_SEPARATOR . $factura->getCliente()->getIdentificacion() . DIRECTORY_SEPARATOR . $archivoName . ".pdf";



        if ($type == "zip") {
            $zip = new \ZipArchive();
            $zipDir = "../web/zip/" . $archivoName . '.zip';

            $zip->open($zipDir, \ZipArchive::CREATE);

            if (file_exists($pathXML)) {
                $zip->addFromString(basename($pathXML), file_get_contents($pathXML));
            }
            if (file_exists($pathPDF)) {
                $zip->addFromString(basename($pathPDF), file_get_contents($pathPDF));
            }

            $zip->close();
            $response = new Response();

            $response->headers->set('Content-Type', 'application/zip');
            $response->headers->set('Content-Disposition', 'attachment; filename="' . basename($zipDir) . '"');
            $response->headers->set('Pragma', "no-cache");
            $response->headers->set('Expires', "0");
            $response->headers->set('Content-Transfer-Encoding', "binary");
            $response->sendHeaders();
            $response->setContent(readfile($zipDir));
            return $response;
        } else if ($type == "pdf") {
            $response = new Response();

            $response->headers->set('Content-Type', 'application/pdf');
            $response->headers->set('Content-Disposition', 'attachment; filename="' . basename($pathPDF) . '"');
            $response->headers->set('Pragma', "no-cache");
            $response->headers->set('Expires', "0");
            $response->headers->set('Content-Transfer-Encoding', "binary");
            $response->sendHeaders();
            $response->setContent(readfile($pathPDF));
            return $response;
        }
    }

    /**
     * Displays a form to create a new Factura entity.
     *
     * @Route("/nueva", name="factura_new")
     * @Method("GET")
     * @Secure(roles="ROLE_EMISOR")
     * @Template()
     */
    public function newAction()
    {
        $em = $this->getDoctrine()->getManager();
        $id = '';
        $em = $this->getDoctrine()->getManager();
        $emisorId = $this->get("security.context")->gettoken()->getuser()->getEmisor()->getId();
        $ConsuFinal = $em->getRepository('FactelBundle:Cliente')->findClienteIdFinal($emisorId, '9999999999999');


        if ($ConsuFinal) {
            foreach ($ConsuFinal as $ConsuFina2) {
                $id = $ConsuFina2->getId();
            }
        }

        $ptoEmision = $em->getRepository('FactelBundle:PtoEmision')->findPtoEmisionEstabEmisorByUsuario($this->get("security.context")->gettoken()->getuser()->getId());
        if ($ptoEmision != null && count($ptoEmision) > 0) {
            return array(
                'pto' => $ptoEmision,
                'cliente' => $id,
            );
        } else {
            throw $this->createNotFoundException('El usuario del sistema no tiene asignado un Punto de Emision.');
        }
    }

    /**
     * @Route("/cargar", name="facturas_load")
     * @Method("GET")
     * @Secure(roles="ROLE_EMISOR_ADMIN")
     * @Template()
     */
    public function cargarFacturaAction()
    {
        $form = $this->createFacturaMasivaForm();
        $em = $this->getDoctrine()->getManager();
        $emisorId = $this->get("security.context")->gettoken()->getuser()->getEmisor()->getId();
        $entities = $em->getRepository('FactelBundle:CargaArchivo')->findBy(array("type" => "FACTURA", "emisor" => $emisorId), array('createdAt' => 'DESC'));
        return array(
            'form' => $form->createView(),
            'entities' => $entities
        );
    }

    /**
     * @Route("/cargar/error/{id}", name="facturas_load_errors")
     * @Method("GET")
     * @Secure(roles="ROLE_EMISOR_ADMIN")
     */
    public function cargarFacturaErrorAction($id)
    {
        $form = $this->createFacturaMasivaForm();
        $em = $this->getDoctrine()->getManager();
        $cargaArchivo = $em->getRepository('FactelBundle:CargaArchivo')->find($id);
        $errors = [];
        $cargaArchivoErrors = $cargaArchivo ? $cargaArchivo->getErrors() : [];
        foreach ($cargaArchivoErrors as $error) {
            $errors[] = $error->getMessage();
        }
        $emisorId = $this->get("security.context")->gettoken()->getuser()->getEmisor()->getId();
        $entities = $em->getRepository('FactelBundle:CargaArchivo')->findBy(array("type" => "FACTURA", "emisor" => $emisorId), array('createdAt' => 'DESC'));
        return $this->render("FactelBundle:Factura:cargarFactura.html.twig", array(
            'form' => $form->createView(),
            'entities' => $entities,
            "errors" => $errors
        ));
    }

    /**
     * Finds and displays a Factura entity.
     *
     * @Route("/{id}", name="factura_show")
     * @Method("GET")
     * @Secure(roles="ROLE_EMISOR")
     * @Template()
     */
    public function showAction($id)
    {
        $em = $this->getDoctrine()->getManager();
        $factura = new Factura();
        $factura = $em->getRepository('FactelBundle:Factura')->findFacturaById($id);
        $archivoName = $factura->getNombreArchivo();
        $claveAcceso = $factura->getClaveAcceso();
        $pathPDF = $factura->getEmisor()->getDirDocAutorizados() . DIRECTORY_SEPARATOR . $factura->getCliente()->getIdentificacion() . DIRECTORY_SEPARATOR . $archivoName . ".pdf";


        if (file_exists($pathPDF)) {
            copy($pathPDF, '/var/www/html/web/upload/pdf/' . $claveAcceso . '.pdf');
        }


        if (!$factura) {
            throw $this->createNotFoundException('Unable to find Factura entity.');
        }

        return array(
            'entity' => $factura,
        );
    }

    /**
     * Displays a form to edit an existing Factura entity.
     *
     * @Route("/{id}/editar", name="factura_edit")
     * @Method("GET")
     * @Secure(roles="ROLE_EMISOR")
     * @Template()
     */
    public function editAction($id)
    {
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('FactelBundle:Factura')->findFacturaById($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Factura entity.');
        }
        if ($entity->getEstado() == "AUTORIZADO" || $entity->getEstado() == "ERROR") {
            $this->get('session')->getFlashBag()->add(
                'notice',
                "Solo puede ser editada la factura en estado: NO AUTORIZADO, DEVUELTA y PROCESANDOSE"
            );
            return $this->redirect($this->generateUrl('factura_show', array('id' => $entity->getId())));
        }
        return array(
            'entity' => $entity,
        );
    }

    /**
     * Creates a form to edit a Factura entity.
     *
     * @param Factura $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createEditForm(Factura $entity)
    {
        $form = $this->createForm(new FacturaType(), $entity, array(
            'action' => $this->generateUrl('factura_update', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }

    /**
     * Deletes a Factura entity.
     *
     * @Route("/{id}", name="factura_delete")
     * @Secure(roles="ROLE_EMISOR")
     * @Method("DELETE")
     */
    public function deleteAction(Request $request, $id)
    {
        $form = $this->createDeleteForm($id);
        $form->handleRequest($request);

        if ($form->isValid()) {
            $em = $this->getDoctrine()->getManager();
            $entity = $em->getRepository('FactelBundle:Factura')->find($id);

            if (!$entity) {
                throw $this->createNotFoundException('Unable to find Factura entity.');
            }

            $em->remove($entity);
            $em->flush();
        }

        return $this->redirect($this->generateUrl('factura'));
    }

    /**
     * Creates a form to delete a Factura entity by id.
     *
     * @param mixed $id The entity id
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createDeleteForm($id)
    {
        return $this->createFormBuilder()
            ->setAction($this->generateUrl('factura_delete', array('id' => $id)))
            ->setMethod('DELETE')
            ->add('submit', 'submit', array('label' => 'Delete'))
            ->getForm()
        ;
    }

    private function claveAcceso($factura, $emisor, $establecimiento, $ptoEmision, $fechaEmision)
    {
        $claveAcceso = str_replace("/", "", $fechaEmision);
        $claveAcceso .= "01";
        $claveAcceso .= $emisor->getRuc();
        $claveAcceso .= $factura->getAmbiente();
        $serie = $establecimiento->getCodigo() . $ptoEmision->getCodigo();
        $claveAcceso .= $serie;
        $claveAcceso .= $factura->getSecuencial();
        $claveAcceso .= "12345678";
        $claveAcceso .= $factura->getTipoEmision();
        $claveAcceso .= $this->modulo11($claveAcceso);

        return $claveAcceso;
    }

    private function modulo11($claveAcceso)
    {
        $multiplos = [2, 3, 4, 5, 6, 7];
        $i = 0;
        $cantidad = strlen($claveAcceso);
        $total = 0;
        while ($cantidad > 0) {
            $total += intval(substr($claveAcceso, $cantidad - 1, 1)) * $multiplos[$i];
            $i++;
            $i = $i % 6;
            $cantidad--;
        }
        $modulo11 = 11 - $total % 11;
        if ($modulo11 == 11) {
            $modulo11 = 0;
        } else if ($modulo11 == 10) {
            $modulo11 = 1;
        }

        return strval($modulo11);
    }

    public function getUploadRootDir()
    {
        // the absolute directory path where uploaded
        // documents should be saved
        return __DIR__ . '/../../../web/upload';
    }

    public function createFacturaMasivaForm()
    {

        $builder = $this->createFormBuilder();
        $builder->setAction($this->generateUrl('factura_create_masivo'));
        $builder->setMethod('POST');

        $builder->add('Facturas', 'file');

        $builder->add('import', 'submit', array(
            'label' => 'Cargar',
            'attr' => array('class' => 'import'),
        ));
        return $builder->getForm();
    }
}
